/** @file   predatoralien.cpp
 * @brief   Implementation of PredatorAlien - class.
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */

#include "predatoralien.h" // class's header file
#include "predatorai.h"
#include "animplayer.h"
#include "warglobals.h"
#include "GfxManager.h"
#include "settings.h"
#include "soundsamples.h"
#include "redrawqueue.h"
#include "www_assert.h"
#include "AnimId.h"
using namespace eng2d;

namespace WeWantWar {



/** Constructor.
 */
PredatorAlien::PredatorAlien() :
  Alien(),
  
  m_spdVec( 0, 0 ),
  m_damagePoints( 22 )
{
  ObjectID::Type oid = ObjectID::TYPE_PREDATORALIEN;
  
  // Set properties
  m_spdVec.y( - Settings::floatObjProp(oid, "spd_walk:") );
  m_damagePoints = Settings::intObjProp(oid, "damagepoints:");
  
  this->setCorrectAnimation( GameAnims::EIdle );
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  this->setArmor( Settings::floatObjProp(oid, "armor:") );
  this->setController( new PredatorAI(this) );
  this->setProperties( GameObject::PROP_GRENADEFORCE );
  

  // Set the collisionpoints
  this->setCollisionPoint( 0, Vec2D(-18,-18) );
  this->setCollisionPoint( 1, Vec2D( 18,-18) );
  this->setCollisionPoint( 2, Vec2D( 18, 18) );
  this->setCollisionPoint( 3, Vec2D(-18, 18) );
  this->addCollisionPoint( Vec2D( 0,-18) );
  this->addCollisionPoint( Vec2D( 0, 18) );
  this->addCollisionPoint( Vec2D(18,  0) );
  this->addCollisionPoint( Vec2D(-18, 0) );
}

/** Destructor.
 */
PredatorAlien::~PredatorAlien()
{
}



/** Updates this alien
 */
void PredatorAlien::update()
{
  // If we're dead, we do nothing.
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  // Are we dying at the moment.
  if ( this->state() == GameObject::STATE_DYING ) {
    // We're dying. If we've shown the dying animation through, we
    // can go to killed-state.
    if ( AnimatedObject::m_animation.paused() == true ) {
      // Animation has been shown through so we can change our state.
      this->state( GameObject::STATE_KILLED );
      this->setCorrectAnimation( GameAnims::EKilled );
      return;
    }
    
  } else {
    // We're not dead nor dying.
    m_pController->update();
    
    if ( m_pController->shoot() ) {
      this->attack();
    }
    
    if ( m_pController->shoot() == 0 && this->getAnimID() == GameAnims::EShoot ) {
      // The alien is no longer attackin player. Set the default animation
      // back
      this->setCorrectAnimation( GameAnims::EWalk );
      
    }

    // If we should walk, we walk
    Vec2D v = m_spdVec;
    v.rotate( this->angle() );
    if ( m_pController->forward() == 1 ) {
      this->move( v );
      this->setCorrectAnimation( GameAnims::EWalk );
      
    } else if ( m_pController->forward() == 2 ) {
      v.scale( 1.4 );
      this->move( v );
      this->setCorrectAnimation( GameAnims::EWalk );

    } else if ( m_pController->forward() == 0 ) {
      if ( this->getAnimID() == GameAnims::EWalk ) {
//        this->setCorrectAnimation( GameAnims::IDLE );
      }
    }
    this->changeAngle( m_pController->turn() );
  }
}



/** Kills this alien.
 */
void PredatorAlien::kill()
{
  if ( this->state() == GameObject::STATE_KILLED ||
       this->state() == GameObject::STATE_DYING ) {
    // This alien is already dying.
    return;
  }
  this->state( GameObject::STATE_DYING );

  const Animation& dyingAnim = GameAnims::findAnimation(
      AnimId::KPredatorAlien, GameAnims::EDying );
  this->setAnimation( dyingAnim, GameAnims::EDying );
  this->setCorrectAnimation( GameAnims::EDying );
  this->makeSound( GameObject::SND_DIE );
}



/** Makes sound
 */
void PredatorAlien::makeSound( GameObject::SoundID id ) const
{
  if ( id == GameObject::SND_PAIN ) {
    int t = ( rand() % 2 < 1 ) ? SMP_ALIEN1_PAIN : SMP_ALIEN1_PAIN2;
    Sound::playSample( t, false );

  } else if ( id == GameObject::SND_DIE ) {
    Sound::playSample( SMP_ALIEN1_DIE, false );
  }
}



/** Hit by bullet
 */
bool PredatorAlien::hitByBullet( Bullet* pB )
{
  bool ret = Alien::hitByBullet( pB );
  if ( ret == true ) {
    if ( pB->iType != Bullet::EFlameThrower ) {
      ParticleBlood* pP = new ParticleBlood( pB->iPosition, pB->velocity(),
                                             12, Color(130,60,10) );
      WarGlobals::pPartManager->addSystem( pP );
    }
    
    AIController* pC = dynamic_cast< AIController* >( this->getController() );
    AIController::BFlag flags = pC->getFlags();
    flags &= ~AIController::WAIT_TARGET;
    pC->setFlags( flags );
    
  }
  return ret;
}



/** Redraws this alien.
 */
void PredatorAlien::redraw( RedrawQueue* pQueue )
{
  if ( this->hidden() == true ) {
    return;
  }
  eng2d::Vec2D p( this->position() );
  int alX = static_cast<int>( p.x() ) - Map::scrollX;
  int alY = static_cast<int>( p.y() ) - Map::scrollY;
  BITMAP* pS = m_animation.currentFrame().asBitmap();
  alX -= pS->w / 2;
  alY -= pS->h / 2;
  pQueue->addRotatedSprite( this->redrawPriority(), alX,alY, itofix( m_angle ),
                            pS );
}



///
/// Getter methods
/// ==============

/** Returns the type of this object.
 */
ObjectID::Type PredatorAlien::objectType() const
{
  return ObjectID::TYPE_PREDATORALIEN;
}



///
/// Private and protected methods
/// =============================

/** Sets the requested animation
 */
void PredatorAlien::setCorrectAnimation( int aAnimId )
{
  if ( this->getAnimID() != aAnimId ) {
    const Animation anim = GameAnims::findAnimation(
        AnimId::KPredatorAlien, aAnimId);
    this->setAnimation( anim, aAnimId );
  }
}



/** Sets the requested animation.
 */
void PredatorAlien::attack()
{
  // We should attack the player. Show blood animation.
  AIController* pAIC = dynamic_cast<AIController*>( this->getController() );
  GameObject* pO = pAIC->getTarget();
  
  WWW_ASSERT( pO != 0 );
  
  //Vec2D plaPos = WarGlobals::pObjTable->pPlayer->position();
  Vec2D plaPos = pO->position();
  Vec2D dirV = plaPos - this->position();
  ParticleBlood* pPB = new ParticleBlood( plaPos, dirV, 10, Color(140,70,30) );
  WarGlobals::pPartManager->addSystem( pPB );
  
  this->setCorrectAnimation( GameAnims::EShoot );
  
  
//  bool died = WarGlobals::pObjTable->pPlayer->causeDamage( m_damagePoints );
  bool died = pO->causeDamage( m_damagePoints );
  if ( died == true ) {
    //WarGlobals::pObjTable->pPlayer->kill();
    pO->kill();
  }
  
  // And we reset the attack-flag.
  m_pController->m_shoot = 0;
}



} // end of namespace

